unit IWGridsWAP;

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing,
  {$ENDIF}
  Classes,
  {$IFDEF VSNET}
  {$ELSE}
  {$IFDEF Linux}QControls, {$ELSE}Controls, {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  IWGridCommon, IWControlWAP, IWTypes, IWWAPTag, IWWAPRenderContext,
  IWBaseInterfaces;

type
  TIWGridCellWAP = class;
  TIWCustomGridWAP = class;

  // This is a TCollectionItem so in the future it can be made visual easier later.
  TIWGridCellWAP = class(TCollectionItem)
  //COLSPAN=Number (columns spanned by the cell)
  //ROWSPAN=Number (rows spanned by the cell)
  protected
//    FAlignment: TAlignment;
    FControl: TIWCustomControlWAP;
//    FFont: TIWFont;
    FHeader: Boolean;
    FHeight: string;
    FText: string;
    FVAlign: TIWGridVAlign;
    FVisible: Boolean;
    FWidth: string;
    FWrap: Boolean;
    FGrid: TIWCustomGridWAP;
    //
    procedure AssignTo(ADest: TPersistent); override;

    procedure SetControl(const AValue: TIWCustomControlWAP);
//    procedure SetFont(const AValue: TIWFont);
    procedure SetHeight(const AValue: string);
    procedure SetWidth(const AValue: string);
  public
    constructor Create(ACollection: TCollection); override;
    destructor Destroy; override;

    function RenderSelf(AGrid: TIWCustomGridWAP; const ARow: Integer;
      const AColumn: Integer; AContext: TIWBaseWAPComponentContext; AText: string = ''): TIWWAPTag;
  published

//    property Alignment: TAlignment read FAlignment write FAlignment;
    property Control: TIWCustomControlWAP read FControl write SetControl;
//    property Font: TIWFont read FFont write SetFont;
    property Header: Boolean read FHeader write FHeader;
    property Height: string read FHeight write SetHeight;
    property Text: string read FText write FText;
    property VAlign: TIWGridVAlign read FVAlign write FVAlign;
    property Visible: Boolean read FVisible write FVisible;
    property Width: string read FWidth write SetWidth;
    property Wrap: Boolean read FWrap write FWrap;
  end;

  TIWGridCellsWAP = class(TOwnedCollection)
  public

    function Add: TIWGridCellWAP;
    // All this is for compatibility with Delphi 5 becouese Owner property is not defined in
    // TOWnedCollection

    function Owner: TPersistent;
  end;

  TIWOnRenderCellWAP = procedure(ACell: TIWGridCellWAP; const ARow: Integer; const AColumn: Integer)
    of object;

  TIWCustomGridWAP = class(TIWCustomControlWAP)
  protected
//    FBorderSize: Integer;
    FCaption: string;
    FCells: array of array of TIWGridCellWAP;
    FCellPadding: Integer;
    FCellSpacing: Integer;
    FFrameBuffer: Integer;
    FLines: TIWGridLines;
    FOnRenderCell: TIWOnRenderCellWAP;
    FUseSize: Boolean;
    FColumnCount: Integer;
    FRowCount: Integer;
    //
    procedure DoRenderCell(ACell: TIWGridCellWAP; const ARow: Integer; const AColumn: Integer);
    procedure RenderCells(AContext: TIWBaseWAPComponentContext; AGridTag: TIWWAPTag); virtual; abstract;
    function IsValidCell(ACell: TIWGridCellWAP): Boolean; virtual; abstract;
    procedure InitControl; override;
  public
    function RenderWap(AContext: TIWBaseWAPComponentContext): TIWWAPTag; override;  published

//    property BorderSize: Integer read FBorderSize write FBorderSize;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property Caption: string read FCaption write FCaption;
    property CellPadding: Integer read FCellPadding write FCellPadding;
    property CellSpacing: Integer read FCellSpacing write FCellSpacing;

 //   property Font;

    property FrameBuffer: Integer read FFrameBuffer write FFrameBuffer;
    property Lines: TIWGridLines read FLines write FLines;
    property OnRenderCell: TIWOnRenderCellWAP read FOnRenderCell write FOnRenderCell;
    property UseSize: Boolean read FUseSize write FUseSize;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWGridWAP.bmp}
  TIWGridWAP = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWGridWAP), 'TIWGridWAP.bmp')]
  {$ENDIF}
  TIWGridWAP = class(TIWCustomGridWAP, IIWSubmitControl)
  protected
    FSubmitParam : String;
    FCellCollection: TIWGridCellsWAP;
    //
    function GetCell(const ARow, AColumn: Integer): TIWGridCellWAP;
    procedure RenderCells(AContext: TIWBaseWAPComponentContext; AGridTag: TIWWAPTag); override;
    procedure SetColumnCount(const AValue: Integer);
    procedure SetRowCount(const AValue: Integer);
    procedure Submit(const AValue: string); override;
    function IsValidCell(ACell: TIWGridCellWAP): Boolean; override;
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    // procedure ClearRendering; override;
    function CellExists(const ARow, AColumn: Integer): Boolean;
    procedure DeleteColumn(const AColumn: Integer);
    procedure DeleteRow(const ARow: Integer);
    function GetSubmitParam : String;

    property Cell[const ARow: Integer; const AColumn: Integer]: TIWGridCellWAP read GetCell;
  published
    property ColumnCount: Integer read FColumnCount write SetColumnCount;
    property RowCount: Integer read FRowCount write SetRowCount;
  end;

implementation

uses
  IWBaseWAPControl, IWBaseWAPInterfaces, SWSystem, SysUtils, IWResourceStrings, SWStrings;

{ TIWGridCellWAP }

procedure TIWGridCellWAP.AssignTo(ADest: TPersistent);
begin
  if ADest is TIWGridCellWAP then begin
    with TIWGridCellWAP(ADest) do begin
//      Alignment := Self.Alignment;
//      Font.Assign(Self.Font);
      Header := Self.Header;
      Height := Self.Height;
      Text := Self.Text;
      VAlign := Self.VAlign;
      Width := Self.Width;
      Wrap := Self.Wrap;
      Visible := Self.Visible;
      Control := Self.Control;
    end;
  end else begin
    inherited;
  end;
end;

constructor TIWGridCellWAP.Create(ACollection: TCollection);
begin
  inherited Create(ACollection);
 // FFont := TIWFont.Create;
  // Copy from Grids font.
  //FFont.Assign(Font);
  FHeight := '0';
  FVisible := True;
  FWidth := '0';
  FControl := nil;
end;

destructor TIWGridCellWAP.Destroy;
begin
//  FreeAndNil(FFont);
  inherited;
end;

function TIWGridCellWAP.RenderSelf(AGrid: TIWCustomGridWAP; const ARow,
  AColumn: Integer; AContext: TIWBaseWAPComponentContext;
  AText: string): TIWWAPTag;
var
  LControlRenderData: TIWWAPTag;
  LVAlign: string;
  LControlContext: TIWBaseWAPComponentContext;
begin
  if not Visible then begin
    Result := TIWWAPTag.CreateTag('TD'); try
      Result.Contents.AddText('&nbsp;');
    except FreeAndNil(Result); raise; end;
  end else begin
    FGrid := AGrid; try
      AGrid.DoRenderCell(Self, ARow, AColumn);
      LControlRenderData := nil;
      if Control <> nil then begin
        LControlContext := TIWBaseWAPComponentContext.Create(Control, AContext.ContainerContext, AContext.PageContext);
        LControlRenderData := Control.RenderWap(LControlContext);
        LControlRenderData.AddParmsList(Control.ExtraTagParams);
        if Assigned(LControlRenderData) then begin
          BaseWAPComponentInterface(Control).DoWAPTag(LControlRenderData);
//          if LControlRenderData.Params.Values['NAME'] = '' then begin
//            LControlRenderData.AddStringParam('NAME', Control.HTMLName);
//          end;
        end;
        AContext.ContainerContext.AddComponent(LControlContext, nil);
      end;
      // VAlign
      LVAlign := '';
      case FVAlign of
        vaMiddle: LVAlign := 'VAlign=middle';
        vaTop: LVAlign := 'VAlign=top';
        vaBottom: LVAlign := 'VAlign=bottom';
        vaBaseline: LVAlign := 'VAlign=baseline';
      end;
      //

      if Length(AText) = 0 then begin
        AText := Text;
      end;

      AText := iif((Trim(AText) = ''){ and (LControlRenderData = nil)}, '&nbsp;', AText);

      Result := TIWWAPTag.CreateTag(iif(Header, 'TH', 'TD')); try
        if AContext.WebApplication.WAPClient.OutputTagType = wotXHTMLMP then begin
          Result.Add(LVAlign);
        end;
        // Always render alignment, default is different or TH and TD
  //      Result.AddStringParam('align', IWGridAlignments[Alignment]);
   //     Result.Add(iif(not Wrap, 'NOWRAP'));
        Result.Add(iif((Width <> '') and (Width <> '0'), 'Width=' + Width));
        Result.Add(iif((Height <> '') and (Height <> '0'), 'Height=' + Height));
//        Result.Add(TIWControl.HTMLColor(BGColor, 'BGColor'));
//        Result.Add(iif(Length(Font.CSSStyle) > 0, 'CLASS=' + Font.CSSStyle));

        // FontToString checks Font.Enabled
{        if Font.CSSStyle = '' then
        begin
           if Result.Contents.AddTagAsObject(Font.FontToStringTag(AText)) = nil then
              Result.Contents.AddText(AText);
        end
        else
        begin
           Result.Contents.AddText(AText);
        end;}
        if (Trim(AText) <> '') and (Control = nil) then begin
//          Result.Contents.AddTagAsObject(Font.FontToStringTag32(AText));
            if AContext.WebApplication.WAPClient.OutputTagType = wotWML then begin
              AText := TIWBaseWAPControl.TextToWML(Atext);
            end;
            // else begin
             // AText := TIWBaseWAPControl.TextToXHTMLMP(Atext);
            //end;
            Result.Contents.AddText(AText)
        end;

        if Assigned(LControlRenderData) then begin
          Result.Contents.AddTagAsObject(LControlRenderData);
        end;
      except FreeAndNil(Result); raise; end;
    finally FGrid := nil; end;
  end;
end;

procedure TIWGridCellWAP.SetHeight(const AValue: string);
var
  s: string;

  procedure InvalidHeight;
  begin
    raise Exception.Create(Format(RSInvalidHeight, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidHeight;
  // Percentage must be >0
  end else if (s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0) then begin
    FHeight := AValue;
  end else if StrToIntDef(s, -1) >= 0 then begin
    FHeight := AValue;
  end else begin
    InvalidHeight;
  end;
end;

procedure TIWGridCellWAP.SetWidth(const AValue: string);
var
  s: string;

  procedure InvalidWidth;
  begin
    raise Exception.Create(Format(RSInvalidWidth, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidWidth;
  // Percentage must be >0
  end else if (s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0) then begin
    FWidth := AValue;
  end else if StrToIntDef(s, -1) >= 0 then begin
    FWidth := AValue;
  end else begin
    InvalidWidth;
  end;
end;

{ TIWGridCellsWAP }

function TIWGridCellsWAP.Add: TIWGridCellWAP;
begin
  Result := TIWGridCellWAP(inherited Add);
end;

function TIWGridCellsWAP.Owner: TPersistent;
begin
  Result := inherited GetOwner;
end;

{ TIWCustomGridWAP }

procedure TIWCustomGridWAP.DoRenderCell(ACell: TIWGridCellWAP; const ARow,
  AColumn: Integer);
begin
  if Assigned(OnRenderCell) then begin
    OnRenderCell(ACell, ARow, AColumn);
  end;
end;

procedure TIWCustomGridWAP.InitControl;
begin
  inherited;
//  FBorderSize := 1;
  FFrameBuffer := 40;
//  FNeedsFormTag := True;
//  FUseSize := True;

  FCaption := '';

  Width := 300;
  Height := 150;
end;

function TIWCustomGridWAP.RenderWAP(
  AContext: TIWBaseWAPComponentContext): TIWWAPTag;
begin
  // If Control property is used.
  Result := TIWWAPTag.CreateTag('TABLE');
  if Acontext.WebApplication.WAPClient.OutputTagType = wotWML then begin
    Result.AddIntegerParam('columns', FColumnCount);
  end;
  try
{    if BorderSize >= 0 then begin
      Result.AddIntegerParam('Border', BorderSize);
    end;
} //   Result.AddColor('BGColor', BGColor);
  if AContext.WebApplication.WAPClient.OutputTagType = wotXHTMLMP then begin
    Result.AddIntegerParam('CELLPADDING', CellPadding);
    Result.AddIntegerParam('CELLSPACING', CellSpacing);
  end;
    if UseSize then begin
      Result.AddIntegerParam('WIDTH', Width);
    end;
    //TODO: Convert ot 3.2
  {  if BorderSize > 0 then begin
      Result.AddColor('BORDERCOLOR', BorderColors.Color);
      Result.AddColor('BORDERCOLORLIGHT', BorderColors.Light);
      Result.AddColor('BORDERCOLORDARK', BorderColors.Dark);
      case Lines of
        tlNone: Result.AddStringParam('RULES', 'none');
        tlRows: Result.AddStringParam('RULES', 'rows');
        tlCols: Result.AddStringParam('RULES', 'cols');
      end;
    // until here.
    end;
}
     {Result.Contents.AddText(iif(Caption, '<CAPTION>' + Caption + '</CAPTION>'));}
{     if Caption <> '' then
     begin
        Result.Contents.AddText('<CAPTION>');
        Result.Contents.AddTagAsObject(WebFont.FontToStringTag32(Caption));
        Result.Contents.AddText('</CAPTION>');
     end;    }
    RenderCells(TIWBaseWAPComponentContext(AContext), Result);
  except FreeAndNil(Result); raise; end;
end;

{ TIWGridWAP }

function TIWGridWAP.CellExists(const ARow, AColumn: Integer): Boolean;
begin
  if (AColumn >= ColumnCount) or (ARow >= RowCount) or (AColumn < 0) or (ARow < 0) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  Result := FCells[ARow, AColumn] <> nil;
end;

procedure TIWGridWAP.DeleteColumn(const AColumn: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AColumn < 0) or (AColumn >= ColumnCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;

  for i := 0 to High(FCells) do begin
    for j := AColumn to High(FCells[i]) - 1 do begin
      if (j = AColumn) and CellExists(i, j) then begin
        FCells[i, j].Free;
      end;
      FCells[i, j] := FCells[i, j + 1];
    end;
  end;

  for i := 0 to High(FCells) do begin
    FCells[i, ColumnCount - 1] := nil;
  end;
  // This will call SetColumnCount
  ColumnCount := ColumnCount - 1;
end;

function TIWGridWAP.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

procedure TIWGridWAP.DeleteRow(const ARow: Integer);
var
  i, j: Integer;
begin
  if (ARow < 0) or (ARow >= RowCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  for i := High(FCells[ARow]) downto 0 do begin
    if CellExists(ARow, i) then begin
      FCellCollection.Delete(FCells[ARow, i].Index);
    end;
  end;
  for i := ARow to High(FCells) - 1 do begin
    for j := 0 to High(FCells[i]) do begin
      FCells[i, j] := FCells[i + 1, j];
    end;
  end;
  for i := 0 to High(FCells[RowCount - 1]) do begin
    FCells[RowCount - 1, i] := nil;
  end;
  // This will call SetRowCount
  RowCount := RowCount - 1;
end;

function TIWGridWAP.GetCell(const ARow, AColumn: Integer): TIWGridCellWAP;
begin
  if not CellExists(ARow, AColumn) then begin
    FCells[ARow, AColumn] := FCellCollection.Add;
    // this will copy current Grid font settings to all cells after they were created
    // Later in OnRenderCell the user program can change these settings by loading different values in
    // GridCell.Font
//    FCells[ARow, AColumn].Font.Assign(Self.Font);
  end;
  Result := FCells[ARow, AColumn];
end;

procedure TIWGridWAP.InitControl;
begin
  inherited;
  FCellCollection := TIWGridCellsWAP.Create(Self, TIWGridCellWAP);
  // Bypass setter routines
  FColumnCount := 1;
  FRowCount := 1;
  SetLength(FCells, 1);
  SetLength(FCells[0], 1);
end;

function TIWGridWAP.IsValidCell(ACell: TIWGridCellWAP): Boolean;
Var
  LCell: TCollectionItem;
begin
  LCell := FCellCollection.FindItemID(ACell.ID);
  result := LCell = ACell;
end;

procedure TIWGridWAP.RenderCells(AContext: TIWBaseWAPComponentContext;
  AGridTag: TIWWAPTag);
var
  i: Integer;
  j: Integer;
begin
  for i := 0 to RowCount - 1 do begin
    with AGridTag.Contents.AddTag('TR') do begin
      for j := 0 to ColumnCount - 1 do begin
        if CellExists(i, j) then begin
          with FCells[i, j] do begin
            if Visible then begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
            end else begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
            end;
          end;
        end else begin
          // Not &nbsp; - only if cell exists but is empty
          with Contents.AddTag('TD') do
            Contents.AddText('');
        end;
      end;
    end;
  end;
end;

procedure TIWGridWAP.SetColumnCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AValue <> FColumnCount) and (AValue > 0) then begin
    if not IsDesignMode then begin
      for i := 0 to High(FCells) do begin
        if AValue < FColumnCount then begin
          for j := ColumnCount - 1 downto AValue do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i,j].Index);
            end;
          end;
        end;
        SetLength(FCells[i], AValue);
        if AValue > FColumnCount then begin
          for j := FColumnCount to AValue - 1 do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    end;
    FColumnCount := AValue;
  end;
end;

procedure TIWGridWAP.SetRowCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AValue <> FRowCount) and (AValue > 0) then begin
    if not IsDesignMode then begin
      if AValue < FRowCount then begin
        for i := FRowCount - 1 downto AValue do begin
          for j := ColumnCount - 1 downto 0 do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i, j].Index);
            end;
          end;
        end;
      end;
      SetLength(FCells, AValue);
      if AValue > FRowCount then begin
        for i := FRowCount to AValue - 1 do begin
          SetLength(FCells[i], FColumnCount);
          for j := 0 to High(FCells[i]) do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    end;
    FRowCount := AValue;
  end;
end;

procedure TIWGridWAP.Submit(const AValue: string);
var
  s: string;
  {LColumn: Integer;
  LRow: Integer;}
begin
  s := AValue;
  {LRow := StrToInt(Fetch(s, '_'));
  LColumn := StrToInt(s);}
end;

procedure TIWGridWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FCellCollection);
  inherited;
end;


procedure TIWGridCellWAP.SetControl(const AValue: TIWCustomControlWAP);
begin
  if AValue <> nil then begin
    if not (csDesigning in AValue.ComponentState) then begin
      AValue.Visible := False;
      AValue.Parent := nil;
    end;

    if AValue is TIWCustomGridWAP then begin
      if FGrid = AValue then begin
        raise Exception.Create(RSCanSetControlToSelf);
      end;
      with TIWCustomGridWAP(AValue) do begin
        if IsValidCell(self) then begin
          raise Exception.Create(RSCanSetControlToSelf);
        end;
      end;
    end;
  end;
  FControl := AValue;
end;

end.
